"use strict";
var config_1 = require('../util/config');
var config_2 = require('../util/config');
var path_1 = require('path');
describe('config', function () {
    describe('generateContext', function () {
        it('should set isWatch true with isWatch true context', function () {
            var context = config_1.generateContext({
                isWatch: true
            });
            expect(context.isWatch).toEqual(true);
        });
        it('should set isWatch false by default', function () {
            var context = config_1.generateContext();
            expect(context.isWatch).toEqual(false);
        });
        it('should set isProd false with isProd false context', function () {
            var context = config_1.generateContext({
                isProd: false
            });
            expect(context.isProd).toEqual(false);
        });
        it('should set default bundler when invalid value', function () {
            var context = config_1.generateContext();
            expect(context.bundler).toEqual('webpack');
        });
        it('should set default bundler when not set', function () {
            var context = config_1.generateContext();
            expect(context.bundler).toEqual('webpack');
        });
        it('should set isProd by default', function () {
            var context = config_1.generateContext();
            expect(context.isProd).toEqual(false);
        });
        it('should create an object when passed nothing', function () {
            var context = config_1.generateContext();
            expect(context).toBeDefined();
        });
        it('should set default prod specific build flag defaults to false', function () {
            var context = config_1.generateContext({
                isProd: false
            });
            expect(context.isProd).toEqual(false);
            expect(context.runAot).toEqual(false);
            expect(context.runMinifyJs).toEqual(false);
            expect(context.runMinifyCss).toEqual(false);
            expect(context.optimizeJs).toEqual(false);
        });
        it('should set default prod specific build flags to true when isProd is true', function () {
            var context = config_1.generateContext({
                isProd: true
            });
            expect(context.isProd).toEqual(true);
            expect(context.runAot).toEqual(true);
            expect(context.runMinifyJs).toEqual(true);
            expect(context.runMinifyCss).toEqual(true);
            expect(context.optimizeJs).toEqual(true);
        });
    });
    describe('replacePathVars', function () {
        it('should interpolated value when string', function () {
            var context = {
                srcDir: 'src',
            };
            var rtn = config_1.replacePathVars(context, '{{SRC}}');
            expect(rtn).toEqual('src');
        });
        it('should interpolated values in string array', function () {
            var context = {
                wwwDir: 'www',
                srcDir: 'src',
            };
            var filePaths = ['{{SRC}}', '{{WWW}}'];
            var rtn = config_1.replacePathVars(context, filePaths);
            expect(rtn).toEqual(['src', 'www']);
        });
        it('should interpolated values in key value pair', function () {
            var context = {
                wwwDir: 'www',
                srcDir: 'src',
            };
            var filePaths = {
                src: '{{SRC}}',
                www: '{{WWW}}'
            };
            var rtn = config_1.replacePathVars(context, filePaths);
            expect(rtn).toEqual({
                src: 'src',
                www: 'www'
            });
        });
    });
    describe('getConfigValue', function () {
        it('should get arg full value', function () {
            config_2.addArgv('--full');
            config_2.addArgv('fullArgValue');
            config_2.addArgv('-s');
            config_2.addArgv('shortArgValue');
            config_2.setProcessEnvVar('ENV_VAR', 'myProcessEnvVar');
            config_2.setAppPackageJsonData({ config: { config_prop: 'myPackageConfigVal' } });
            var val = config_1.getConfigValue(context, '--full', '-s', 'ENV_VAR', 'config_prop', 'defaultValue');
            expect(val).toEqual('fullArgValue');
        });
        it('should get arg short value', function () {
            config_2.addArgv('-s');
            config_2.addArgv('shortArgValue');
            config_2.setProcessEnvVar('ENV_VAR', 'myProcessEnvVar');
            config_2.setAppPackageJsonData({ config: { config_prop: 'myPackageConfigVal' } });
            var val = config_1.getConfigValue(context, '--full', '-s', 'ENV_VAR', 'config_prop', 'defaultValue');
            expect(val).toEqual('shortArgValue');
        });
        it('should get envVar value', function () {
            config_2.setProcessEnvVar('ENV_VAR', 'myProcessEnvVar');
            config_2.setAppPackageJsonData({ config: { config_prop: 'myPackageConfigVal' } });
            var val = config_1.getConfigValue(context, '--full', '-s', 'ENV_VAR', 'config_prop', 'defaultValue');
            expect(val).toEqual('myProcessEnvVar');
        });
        it('should get package.json config value', function () {
            config_2.setAppPackageJsonData({ config: { config_prop: 'myPackageConfigVal' } });
            var val = config_1.getConfigValue(context, '--full', '-s', 'ENV_VAR', 'config_prop', 'defaultValue');
            expect(val).toEqual('myPackageConfigVal');
        });
        it('should get default value', function () {
            var val = config_1.getConfigValue(context, '--full', '-s', 'ENV_VAR', 'config_prop', 'defaultValue');
            expect(val).toEqual('defaultValue');
        });
    });
    describe('bundlerStrategy', function () {
        it('should get rollup by full arg', function () {
            config_2.addArgv('--rollup');
            config_2.addArgv('my.rollup.confg.js');
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get rollup by short arg', function () {
            config_2.addArgv('-r');
            config_2.addArgv('my.rollup.confg.js');
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get rollup by bundler arg', function () {
            config_2.addArgv('--bundler');
            config_2.addArgv('rollup');
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get rollup by env var', function () {
            config_2.setProcessEnv({
                ionic_bundler: 'rollup'
            });
            config_2.setAppPackageJsonData({ config: { ionic_bundler: 'rollup' } });
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get rollup by package.json config', function () {
            config_2.setAppPackageJsonData({ config: { ionic_bundler: 'rollup' } });
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get webpack with invalid env var', function () {
            config_2.setProcessEnv({
                ionic_bundler: 'bobsBundler'
            });
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('webpack');
        });
        it('should get rollup by env var', function () {
            config_2.setProcessEnv({
                ionic_bundler: 'rollup'
            });
            config_2.setAppPackageJsonData({ config: { ionic_bundler: 'rollup' } });
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get rollup by package.json config', function () {
            config_2.setAppPackageJsonData({ config: { ionic_bundler: 'rollup' } });
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('rollup');
        });
        it('should get webpack by default', function () {
            var bundler = config_1.bundlerStrategy(context);
            expect(bundler).toEqual('webpack');
        });
    });
    describe('getUserConfigFile', function () {
        it('should get config from package.json config', function () {
            config_2.setAppPackageJsonData({
                config: { ionic_config: 'myconfig.js' }
            });
            var userConfigFile = null;
            var context = { rootDir: process.cwd() };
            var taskInfo = { fullArg: '--full', shortArg: '-s', defaultConfigFile: 'default.config.js', envVar: 'IONIC_CONFIG', packageConfig: 'ionic_config' };
            var rtn = config_1.getUserConfigFile(context, taskInfo, userConfigFile);
            expect(rtn).toEqual(path_1.resolve('myconfig.js'));
        });
        it('should get config from env var', function () {
            config_2.setProcessEnv({
                IONIC_CONFIG: 'myconfig.js'
            });
            var userConfigFile = null;
            var context = { rootDir: process.cwd() };
            var taskInfo = { fullArg: '--full', shortArg: '-s', defaultConfigFile: 'default.config.js', envVar: 'IONIC_CONFIG', packageConfig: 'ionic_config' };
            var rtn = config_1.getUserConfigFile(context, taskInfo, userConfigFile);
            expect(rtn).toEqual(path_1.resolve('myconfig.js'));
        });
        it('should get config from short arg', function () {
            config_2.addArgv('-s');
            config_2.addArgv('myconfig.js');
            var userConfigFile = null;
            var context = { rootDir: process.cwd() };
            var taskInfo = { fullArg: '--full', shortArg: '-s', defaultConfigFile: 'default.config.js', envVar: 'IONIC_CONFIG', packageConfig: 'ionic_config' };
            var rtn = config_1.getUserConfigFile(context, taskInfo, userConfigFile);
            expect(rtn).toEqual(path_1.resolve('myconfig.js'));
        });
        it('should get config from full arg', function () {
            config_2.addArgv('--full');
            config_2.addArgv('myconfig.js');
            var userConfigFile = null;
            var context = { rootDir: process.cwd() };
            var taskInfo = { fullArg: '--full', shortArg: '-s', defaultConfigFile: 'default.config.js', envVar: 'IONIC_CONFIG', packageConfig: 'ionic_config' };
            var rtn = config_1.getUserConfigFile(context, taskInfo, userConfigFile);
            expect(rtn).toEqual(path_1.resolve('myconfig.js'));
        });
        it('should get userConfigFile', function () {
            var userConfigFile = 'myconfig.js';
            var context = { rootDir: process.cwd() };
            var taskInfo = { fullArg: '--full', shortArg: '-s', defaultConfigFile: 'default.config.js', envVar: 'IONIC_CONFIG', packageConfig: 'ionic_config' };
            var rtn = config_1.getUserConfigFile(context, taskInfo, userConfigFile);
            expect(rtn).toEqual(path_1.resolve('myconfig.js'));
        });
        it('should not get a user config', function () {
            var userConfigFile = null;
            var context = { rootDir: process.cwd() };
            var taskInfo = { fullArg: '--full', shortArg: '-s', defaultConfigFile: 'default.config.js', envVar: 'IONIC_CONFIG', packageConfig: 'ionic_config' };
            var rtn = config_1.getUserConfigFile(context, taskInfo, userConfigFile);
            expect(rtn).toEqual(null);
        });
    });
    describe('hasArg function', function () {
        it('should return false when a match is not found', function () {
            var result = config_1.hasArg('--full', '-f');
            expect(result).toBeFalsy();
        });
        it('should match on a fullname arg', function () {
            config_2.addArgv('--full');
            var result = config_1.hasArg('--full');
            expect(result).toBeTruthy();
        });
        it('should match on a shortname arg', function () {
            config_2.addArgv('-f');
            var result = config_1.hasArg('--full', '-f');
            expect(result).toBeTruthy();
        });
        it('should compare fullnames as case insensitive', function () {
            config_2.addArgv('--full');
            config_2.addArgv('--TEST');
            var result = config_1.hasArg('--Full');
            var result2 = config_1.hasArg('--test');
            expect(result).toBeTruthy();
            expect(result2).toBeTruthy();
        });
        it('should compare shortnames as case insensitive', function () {
            config_2.addArgv('-f');
            config_2.addArgv('-T');
            var result = config_1.hasArg('-F');
            var result2 = config_1.hasArg('-t');
            expect(result).toBeTruthy();
            expect(result2).toBeTruthy();
        });
    });
    var context;
    beforeEach(function () {
        config_2.setProcessArgs(['node', 'ionic-app-scripts']);
        config_2.setProcessEnv({});
        config_2.setCwd('');
        config_2.setAppPackageJsonData(null);
        context = config_1.generateContext({});
    });
});
