import { EventEmitter, Injectable, Optional } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { Config } from '../../config/config';
import { isNav, DIRECTION_FORWARD, DIRECTION_BACK } from '../../navigation/nav-util';
import { Platform } from '../../platform/platform';
import { MenuController } from '../menu/menu-controller';
/**
 * @name App
 * @description
 * App is a utility class used in Ionic to get information about various aspects of an app
 */
export class App {
    constructor(_config, _platform, _menuCtrl) {
        this._config = _config;
        this._platform = _platform;
        this._menuCtrl = _menuCtrl;
        this._disTime = 0;
        this._scrollTime = 0;
        this._title = '';
        this._titleSrv = new Title();
        this._rootNav = null;
        /**
         * @private
         */
        this.viewDidLoad = new EventEmitter();
        /**
         * @private
         */
        this.viewWillEnter = new EventEmitter();
        /**
         * @private
         */
        this.viewDidEnter = new EventEmitter();
        /**
         * @private
         */
        this.viewWillLeave = new EventEmitter();
        /**
         * @private
         */
        this.viewDidLeave = new EventEmitter();
        /**
         * @private
         */
        this.viewWillUnload = new EventEmitter();
        // listen for hardware back button events
        // register this back button action with a default priority
        _platform.registerBackButtonAction(this.goBack.bind(this));
        this._disableScrollAssist = _config.getBoolean('disableScrollAssist', false);
        (void 0) /* runInDev */;
    }
    /**
     * Sets the document title.
     * @param {string} val  Value to set the document title to.
     */
    setTitle(val) {
        if (val !== this._title) {
            this._title = val;
            this._titleSrv.setTitle(val);
        }
    }
    /**
     * @private
     */
    setElementClass(className, isAdd) {
        this._appRoot.setElementClass(className, isAdd);
    }
    /**
     * @private
     * Sets if the app is currently enabled or not, meaning if it's
     * available to accept new user commands. For example, this is set to `false`
     * while views transition, a modal slides up, an action-sheet
     * slides up, etc. After the transition completes it is set back to `true`.
     * @param {boolean} isEnabled `true` for enabled, `false` for disabled
     * @param {number} duration  When `isEnabled` is set to `false`, this argument
     * is used to set the maximum number of milliseconds that app will wait until
     * it will automatically enable the app again. It's basically a fallback incase
     * something goes wrong during a transition and the app wasn't re-enabled correctly.
     */
    setEnabled(isEnabled, duration = 700) {
        this._disTime = (isEnabled ? 0 : Date.now() + duration);
        if (this._clickBlock) {
            if (isEnabled) {
                // disable the click block if it's enabled, or the duration is tiny
                this._clickBlock.activate(false, CLICK_BLOCK_BUFFER_IN_MILLIS);
            }
            else {
                // show the click block for duration + some number
                this._clickBlock.activate(true, duration + CLICK_BLOCK_BUFFER_IN_MILLIS);
            }
        }
    }
    /**
     * @private
     * Toggles whether an application can be scrolled
     * @param {boolean} disableScroll when set to `false`, the application's
     * scrolling is enabled. When set to `true`, scrolling is disabled.
     */
    _setDisableScroll(disableScroll) {
        if (this._disableScrollAssist) {
            this._appRoot._disableScroll(disableScroll);
        }
    }
    /**
     * @private
     * Boolean if the app is actively enabled or not.
     * @return {boolean}
     */
    isEnabled() {
        const disTime = this._disTime;
        if (disTime === 0) {
            return true;
        }
        return (disTime < Date.now());
    }
    /**
     * @private
     */
    setScrolling() {
        this._scrollTime = Date.now() + ACTIVE_SCROLLING_TIME;
    }
    /**
     * Boolean if the app is actively scrolling or not.
     * @return {boolean} returns true or false
     */
    isScrolling() {
        const scrollTime = this._scrollTime;
        if (scrollTime === 0) {
            return false;
        }
        if (scrollTime < Date.now()) {
            this._scrollTime = 0;
            return false;
        }
        return true;
    }
    /**
     * @private
     */
    getActiveNav() {
        const portal = this._appRoot._getPortal(MODAL);
        if (portal.length() > 0) {
            return findTopNav(portal);
        }
        return findTopNav(this._rootNav || null);
    }
    /**
     * @return {NavController} Returns the root NavController
     */
    getRootNav() {
        return this._rootNav;
    }
    /**
     * @private
     */
    _setRootNav(nav) {
        this._rootNav = nav;
    }
    /**
     * @private
     */
    present(enteringView, opts, appPortal) {
        const portal = this._appRoot._getPortal(appPortal);
        enteringView._setNav(portal);
        opts.keyboardClose = false;
        opts.direction = DIRECTION_FORWARD;
        if (!opts.animation) {
            opts.animation = enteringView.getTransitionName(DIRECTION_FORWARD);
        }
        enteringView.setLeavingOpts({
            keyboardClose: false,
            direction: DIRECTION_BACK,
            animation: enteringView.getTransitionName(DIRECTION_BACK),
            ev: opts.ev
        });
        return portal.insertPages(-1, [enteringView], opts);
    }
    /**
     * @private
     */
    goBack() {
        if (this._menuCtrl && this._menuCtrl.isOpen()) {
            return this._menuCtrl.close();
        }
        const navPromise = this.navPop();
        if (navPromise === null) {
            // no views to go back to
            // let's exit the app
            if (this._config.getBoolean('navExitApp', true)) {
                (void 0) /* console.debug */;
                this._platform.exitApp();
            }
        }
        return navPromise;
    }
    /**
     * @private
     */
    navPop() {
        if (!this._rootNav || !this.isEnabled()) {
            return Promise.resolve();
        }
        // If there are any alert/actionsheet open, let's do nothing
        const portal = this._appRoot._getPortal(DEFAULT);
        if (portal.length() > 0) {
            return Promise.resolve();
        }
        // next get the active nav, check itself and climb up all
        // of its parent navs until it finds a nav that can pop
        return recursivePop(this.getActiveNav());
    }
}
App.decorators = [
    { type: Injectable },
];
/** @nocollapse */
App.ctorParameters = [
    { type: Config, },
    { type: Platform, },
    { type: MenuController, decorators: [{ type: Optional },] },
];
function recursivePop(nav) {
    if (!nav) {
        return null;
    }
    if (isNav(nav)) {
        var len = nav.length();
        if (len > 1 || (nav._isPortal && len > 0)) {
            // this nav controller has more than one view
            // pop the current view on this nav and we're done here
            (void 0) /* console.debug */;
            return nav.pop();
        }
    }
    // try again using the parent nav (if there is one)
    return recursivePop(nav.parent);
}
function findTopNav(nav) {
    var activeChildNav;
    while (nav) {
        activeChildNav = nav.getActiveChildNav();
        if (!activeChildNav) {
            break;
        }
        nav = activeChildNav;
    }
    return nav;
}
const DEFAULT = 0; // AppPortal.DEFAULT
const MODAL = 1; // AppPortal.MODAL
const ACTIVE_SCROLLING_TIME = 100;
const CLICK_BLOCK_BUFFER_IN_MILLIS = 64;
//# sourceMappingURL=app.js.map